/**
 * \file: grl_bitmap_decoder_core.c
 *
 * version: $Id: grl_bitmap_decoder_core.c,v 1.24 2010/01/14 13:43:29 tkniep Exp $
 *
 * This file implements the low-level functions of the Bitmap Decoder Core
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdint.h>

#include "svg_bitmap_decoder.h"
#include "grl_bitmap_decoder_util.h"
#include "grl_bitmap_decoder_core.h"
#include "grl_bitmap_decoder_conf.h"

/*******************************************************************************************
 *   Global variables
 *******************************************************************************************/

/** 
 * Global error (used if an error occurs when no context
 * is available
 */
static SVGError g_bmpdec_lasterror = SVG_NO_ERROR;

/**
 * Global variable for storing the number of created contexts.
 * Access is protected with the lock g_bmpdec_ctx_lock to avoid
 * problems when multiple contexts are created or destroyed 
 * simultaneously. 
 */
static SVGUint32 g_bmpdec_ctx_cnt = 0;

/**
 * Global variable for storing the pointers to the created contexts.
 * Storage is needed for cleanly shutting down e.g. if the application 
 * has "forgotten" the pointer.
 */
static SVGContextBmpDec **g_bmpdec_ctx_list = NULL;

/**
 * Global ID for the lock which is used to protect access to
 * g_bmpdec_ctx_cnt.
 */
static GRL_lock_id g_bmpdec_ctx_lock = 0;

/**
 * Global function table containing pointers to the decoder-specific
 * functions for decoding a bitmap.
 */
static GRL_BMPDEC_DRAW_IMAGE_FUNCS
		grl_bmpdec_draw_image_funcs[SVG_BMPDEC_ENCODING_MAX_VALUE];

/**
 * Global function table containing pointers to the decoder-specific
 * functions for getting image info.
 */
static GRL_BMPDEC_GET_IMAGE_INFO_FUNCS
		grl_bmpdec_get_image_info_funcs[SVG_BMPDEC_ENCODING_MAX_VALUE];

/*******************************************************************************************
 *   static function declarations
 *******************************************************************************************/

/**
 * Main function of decoder task.
 * The decoder task is used for asynchronous decoding of bitmaps.
 * It is created by the function GRL_BMPDEC_draw_image_async.
 *
 * \param[in]   exinf           Pointer to Bitmap Decoder context
 */

static void GRL_BMPDEC_dec_tsk(void* exinf);

/*******************************************************************************************
 *   Function Implementations
 *   (documentation is in header file)
 *******************************************************************************************/

SVGError GRL_BMPDEC_draw_image_dummy(GRL_BMPDEC_dec_info *info)
{
	/* Touch value because Lint will complain if it is not referenced */
	info = info;

	SVG_BMP_W("SVG_INVALID_OPERATION IN GRL_BMPDEC_DRAW_IMAGE_DUMMY");
	return SVG_INVALID_OPERATION;
}

SVGError GRL_BMPDEC_get_image_info_dummy(const SVGContextBmpDec *ctx,
		const SVGImage *image, SVGImageInfo *image_info)
{
	/* Touch values because Lint will complain if it is not referenced */
	ctx = ctx;
	image = image;
	image_info = image_info;

	SVG_BMP_W("SVG_INVALID_OPERATION IN GRL_BMPDEC_GET_IMAGE_INFO_DUMMY");
	return SVG_INVALID_OPERATION;
}

SVGError GRL_BMPDEC_init_tables(void)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGBmpDecImageEncoding cnt = SVG_BMPDEC_UNKNOWN_ENCODING;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_INIT_TABLES");

	/* Init the decoding functions with dummy function */
	for (cnt = SVG_BMPDEC_UNKNOWN_ENCODING; cnt < SVG_BMPDEC_ENCODING_MAX_VALUE; cnt++)
	{
		grl_bmpdec_draw_image_funcs[cnt] = GRL_BMPDEC_draw_image_dummy;
	}

	/* Init the image info functions with dummy function */
	for (cnt = SVG_BMPDEC_UNKNOWN_ENCODING; cnt < SVG_BMPDEC_ENCODING_MAX_VALUE; cnt++)
	{
		grl_bmpdec_get_image_info_funcs[cnt] = GRL_BMPDEC_get_image_info_dummy;
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_INIT_TABLES");

	return ret_err;
}

extern SVGError GRL_BMPDEC_draw_image_PNG(GRL_BMPDEC_dec_info *p_decode_info);
extern SVGError GRL_BMPDEC_get_image_info_PNG(const SVGContextBmpDec *ctx,
		const SVGImage *p_image, SVGImageInfo *image_info);

extern SVGError GRL_BMPDEC_draw_image_JPEG(GRL_BMPDEC_dec_info *p_decode_info);
extern SVGError GRL_BMPDEC_get_image_info_JPEG(const SVGContextBmpDec *ctx,
		const SVGImage *p_image, SVGImageInfo *image_info);

extern SVGError GRL_BMPDEC_draw_image_FGS(GRL_BMPDEC_dec_info *p_decode_info);
extern SVGError GRL_BMPDEC_get_image_info_FGS(const SVGContextBmpDec *ctx,
        const SVGImage *p_image, SVGImageInfo *image_info);

SVGError GRL_BMPDEC_reg_decoder(void)
{
	SVGError ret_err = SVG_NO_ERROR;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_REG_DECODER");

	grl_bmpdec_draw_image_funcs[SVG_BMPDEC_JPEG_ENCODING]
			= GRL_BMPDEC_draw_image_JPEG;
	grl_bmpdec_get_image_info_funcs[SVG_BMPDEC_JPEG_ENCODING]
			= GRL_BMPDEC_get_image_info_JPEG;

	grl_bmpdec_draw_image_funcs[SVG_BMPDEC_PNG_ENCODING]
			= GRL_BMPDEC_draw_image_PNG;
	grl_bmpdec_get_image_info_funcs[SVG_BMPDEC_PNG_ENCODING]
			= GRL_BMPDEC_get_image_info_PNG;

	grl_bmpdec_draw_image_funcs[SVG_BMPDEC_PNG_ENCODING_WITH_PLTE]
			= GRL_BMPDEC_draw_image_PNG;
	grl_bmpdec_get_image_info_funcs[SVG_BMPDEC_PNG_ENCODING_WITH_PLTE]
			= GRL_BMPDEC_get_image_info_PNG;

	grl_bmpdec_draw_image_funcs[SVG_BMPDEC_FGS_ENCODING]
	            = GRL_BMPDEC_draw_image_FGS;
	grl_bmpdec_get_image_info_funcs[SVG_BMPDEC_FGS_ENCODING]
	            = GRL_BMPDEC_get_image_info_FGS;

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_REG_DECODER");

	return ret_err;
}

SVGError GRL_BMPDEC_cre_ctx(SVGContextBmpDec **new_ctx)
{

	SVGError ret_err = SVG_NO_ERROR;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};
	SVGUint32 ctx_num = 0;
	SVGUint32 queue_size = 0;
	SVGChar dsname[GRL_BMPDEC_DSNAME_LEN] = GRL_BMPDEC_LCK_NM_BSY;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CRE_CTX");

	GRL_BMPDEC_get_conf(&config);
	ctx_num = GRL_BMPDEC_clst_get_num();

	if (ctx_num < config.ctx_max)
	{
		/* Allocate memory for context from 1D resources */

		(*new_ctx) = (SVGContextBmpDec *) GRL_malloc_1D_resource(
				sizeof(SVGContextBmpDec));
		if (*new_ctx == NULL)
		{
			ret_err = SVG_OUT_OF_MEMORY;
			SVG_BMP_E("SVG_OUT_OF_MEMORY IN GRL_BMPDEC_CRE_CTX");
		} else
		{
			/* Initialize context fields */
			memset(*new_ctx, 0, sizeof(SVGContextBmpDec));

			(*new_ctx)->lastError = SVG_NO_ERROR;
			(*new_ctx)->dec_tsk_id = 0;
			(*new_ctx)->queue_running = SVG_FALSE;
			(*new_ctx)->rgb_to_bgr    = config.rgb_to_bgr;

			/* Allocate memory for decoding info lists (2 lists) */
			queue_size = (SVGUint32) config.queue_size
					* sizeof(GRL_BMPDEC_dec_info);

			(*new_ctx)->p_decoding_info[0]
					= (GRL_BMPDEC_dec_info*) GRL_malloc_1D_resource(
							(SVGUint32) config.queue_size
									* sizeof(GRL_BMPDEC_dec_info));
			if ((*new_ctx)->p_decoding_info[0] == NULL)
			{
				ret_err = SVG_OUT_OF_MEMORY;
				SVG_BMP_E("SVG_OUT_OF_MEMORY IN GRL_BMPDEC_CRE_CTX");
				/* Free previously allocated resources */
				GRL_free_1D_resource(*new_ctx);
			} else
			{
				memset((*new_ctx)->p_decoding_info[0], 0, queue_size);

				(*new_ctx)->p_decoding_info[1]
						= (GRL_BMPDEC_dec_info*) GRL_malloc_1D_resource(
								(SVGUint32) config.queue_size
										* sizeof(GRL_BMPDEC_dec_info));

				if ((*new_ctx)->p_decoding_info[1] == NULL)
				{
					ret_err = SVG_OUT_OF_MEMORY;
					SVG_BMP_E("SVG_OUT_OF_MEMORY IN GRL_BMPDEC_CRE_CTX");

					/* Free previously allocated resources */
					GRL_free_1D_resource((*new_ctx)->p_decoding_info[0]);
					(*new_ctx)->p_decoding_info[0] = NULL;
					(*new_ctx)->p_decoding_info[1] = NULL;

					GRL_free_1D_resource(*new_ctx);
				} else
				{
					memset((*new_ctx)->p_decoding_info[1], 0, queue_size);

					/* Set the write and read queue pointers to the newly created lists */
					(*new_ctx)->p_write_queue = (*new_ctx)->p_decoding_info[0];
					(*new_ctx)->p_read_queue = (*new_ctx)->p_decoding_info[1];
				}
			}
		}

		if (SVG_NO_ERROR == ret_err)
		{
			ret_err = GRL_BMPDEC_clst_add((*new_ctx));
			if (SVG_NO_ERROR != ret_err)
			{
				SVG_BMP_E("ERROR IN GRL_BMPDEC_CRE_CTX");
				SVG_BMP_E("Could not add new context to internal list, creating context aborted.");


				/* Free previously allocated resources */
				GRL_free_1D_resource((*new_ctx)->p_decoding_info[0]);
				(*new_ctx)->p_decoding_info[0] = NULL;
				GRL_free_1D_resource((*new_ctx)->p_decoding_info[1]);
				(*new_ctx)->p_decoding_info[1] = NULL;

				GRL_free_1D_resource(*new_ctx);
				(*new_ctx) = NULL;
			}
		}

		if (SVG_NO_ERROR == ret_err)
		{
			/* Create lock for protecting the context when it's busy */
			ret_err = GRL_cre_lock(&((*new_ctx)->busy_lock), dsname);
			if (SVG_NO_ERROR != ret_err)
			{
				SVG_BMP_E("ERROR IN GRL_BMPDEC_CRE_CTX");
				SVG_BMP_E("Cannot create context busy lock, context will not be created.");

				/* Free previously allocated resources */
				GRL_free_1D_resource((*new_ctx)->p_decoding_info[0]);
				(*new_ctx)->p_decoding_info[0] = NULL;
				GRL_free_1D_resource((*new_ctx)->p_decoding_info[1]);
				(*new_ctx)->p_decoding_info[1] = NULL;

				GRL_free_1D_resource(*new_ctx);
				(*new_ctx) = NULL;
			}
		}
	} else
	{
		/* Maximum number of contexts reached */
		ret_err = SVG_BMPDEC_CTX_MAX;
		SVG_BMP_E("SVG_BMPDEC_CTX_MAX IN GRL_BMPDEC_CRE_CTX");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CRE_CTX");
	return ret_err;
}

SVGError GRL_BMPDEC_dty_ctx(SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	int ercd = 0;

	GRL_BMPDEC_SET_FLG( GRL_BMPDEC_FLG_PTN_ABORT );

	ret_err = GRL_wai_lock_tmo(ctx->busy_lock, GRL_BMPDEC_LOCK_TMO );
	if (SVG_NO_ERROR != ret_err)
	{
		/* Decoder task hangs, kill it */
		ret_err = SVG_TIMEOUT_ERROR;
		SVG_BMP_E("SVG_TIMEOUT_ERROR IN GRL_BMPDEC_DTY_CTX");

		if (ctx->dec_tsk_id > 0)
		{

			GRL_BMPDEC_TER_TSK( ctx->dec_tsk_id, ercd );

			if (0 != ercd)
			{
				/* Could not delete the task */
				ret_err = SVG_BMPDEC_OS_ERROR;
				SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DTY_CTX");
			}
		}
	}
	if (ctx->dec_tsk_id > 0)
	{
		/* Delete event flag */
		ercd = GRL_BMPDEC_DEL_FLG( ctx->dec_flg );

		if (0 > ercd)
		{
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DTY_CTX");
		}

		/* wait for thread*/
		ret_err = pthread_join(ctx->dec_tsk_id, NULL);
		if (SVG_NO_ERROR != ret_err)
		{
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DTY_CTX");
		}
	}
	/*Signal lock, in tkernel delete busy lock is not a problem but not in posix!*/
	ret_err = GRL_sig_lock(ctx->busy_lock);
	if (SVG_NO_ERROR != ret_err)
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DTY_CTX");
	}
	/* Delete lock */
	ret_err = GRL_del_lock(ctx->busy_lock);
	if (SVG_NO_ERROR != ret_err)
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DTY_CTX");
	}

	/* Unregister context */
	ret_err = GRL_BMPDEC_clst_del(ctx);
	if (ret_err != SVG_NO_ERROR)
	{
		ret_err = SVG_BMPDEC_INTERNAL_ERROR;
		SVG_BMP_E("SVG_BMPDEC_INTERNAL_ERROR IN GRL_BMPDEC_DTY_CTX");
	}

	/* Free 1D resources from decoding info */
	GRL_free_1D_resource(ctx->p_decoding_info[0]);

	GRL_free_1D_resource(ctx->p_decoding_info[1]);

	/* Free 1D resources from context */
	GRL_free_1D_resource(ctx);

	return ret_err;
}


SVGError GRL_BMPDEC_check_mode_mem(const SVGImage *image)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGUint32 size = 0;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CHECK_MODE_MEM");

	/* Check destination pointer */
	if (NULL == image->dest.dest_mem.dest_mem)
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_MODE_MEM");
	}

	/* Check destination size */
	size = (SVGUint32) (image->clip.width * image->clip.height
			* GRL_BMPDEC_DEPTH_ARGB8888);
	if ((SVG_NO_ERROR == ret_err) && (image->dest.dest_mem.dest_size < size))
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_MODE_MEM");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CHECK_MODE_MEM");

	return ret_err;
}

SVGError GRL_BMPDEC_check_img_basic(const SVGImage *image)
{
	SVGError ret_err = SVG_NO_ERROR;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CHECK_IMG_BASIC");

	/* Check encoding type */
	if ((SVG_BMPDEC_UNKNOWN_ENCODING >= image->type)
			|| (SVG_BMPDEC_ENCODING_MAX_VALUE <= image->type))
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_IMG_BASIC");
	}

	/* Check pointer and size of compressed image */
	if ((SVG_NO_ERROR == ret_err) && ((NULL == image->datap) || (0
			== image->data_size)))
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_IMG_BASIC");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CHECK_IMG_BASIC");
	return ret_err;
}

SVGError GRL_BMPDEC_check_img(const SVGImage *image)
{
	SVGError ret_err = SVG_NO_ERROR;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CHECK_IMG");

	/* Carry out the basic checks */
	ret_err = GRL_BMPDEC_check_img_basic(image);

	/* Check decoding mode */
	if (SVG_NO_ERROR == ret_err)
	{
		switch (image->decode_mode)
		{
		case SVG_BMPDEC_MODE_MEMORY:
		{
			ret_err = GRL_BMPDEC_check_mode_mem(image);
			break;
		}
		default:
		{
			ret_err = SVG_INVALID_VALUE;
			SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_IMG");
		}
		}
	}

	/* Check width and height */
	if ((SVG_NO_ERROR == ret_err) && ((0 == image->width) || (0
			== image->height)))
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_CHECK_IMG");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CHECK_IMG");

	return ret_err;
}

SVGError GRL_BMPDEC_get_image_info(SVGContextBmpDec *ctx,
									const SVGImage *image,
									SVGImageInfo *image_info)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGBmpDecImageEncoding idx = SVG_BMPDEC_UNKNOWN_ENCODING;
	ctx = ctx;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_GET_IMAGE_INFO");

	idx = image->type;

	if ((SVG_BMPDEC_UNKNOWN_ENCODING < idx) &&
		(SVG_BMPDEC_ENCODING_MAX_VALUE > idx))
	{
		/* Use image type as index to look up correct image info function */
		ret_err = grl_bmpdec_get_image_info_funcs[idx](ctx, image, image_info);
	} else
	{
		ret_err = SVG_INVALID_VALUE;
		SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_GET_IMAGE_INFO");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_GET_IMAGE_INFO");

	return ret_err;
}

SVGError GRL_BMPDEC_get_formats(SVGBmpDecFormats *formats)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGUint32 cnt = 0;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_GET_FORMATS");

	/* Init to all decoders unsupported */
	*formats = 0;

	/* Use the registered draw functions to check whether a decoder was registered */
	for (cnt = SVG_BMPDEC_UNKNOWN_ENCODING + 1; cnt
			< SVG_BMPDEC_ENCODING_MAX_VALUE; cnt++)
	{
		if (grl_bmpdec_draw_image_funcs[cnt] != GRL_BMPDEC_draw_image_dummy)
		{
			/* Entry is different from the default one -> decoder is available */
			*formats |= ((SVGBmpDecFormats) 0x1 << cnt);
		}
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_GET_FORMATS");

	return ret_err;
}

SVGError GRL_BMPDEC_draw_image_sync(SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGError dec_err = SVG_NO_ERROR;
	SVGInt32 cnt = 0;
	GRL_BMPDEC_dec_info *p_dec_info = NULL;
	SVGBmpDecImageEncoding idx = SVG_BMPDEC_UNKNOWN_ENCODING;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_DRAW_IMAGE_SYNC");

	/*
	 * Call the decoder-specific decoding function for each
	 * queued request.
	 */
	for (cnt = 0; cnt < ctx->write_info_cnt; cnt++)
	{
		p_dec_info = &ctx->p_write_queue[cnt];
		p_dec_info->revert = ctx->rgb_to_bgr;
		idx = p_dec_info->image.type;

		if ((SVG_BMPDEC_UNKNOWN_ENCODING < idx)
				&& (SVG_BMPDEC_ENCODING_MAX_VALUE > idx))
		{
			/* Use image type as index to look up correct decoding function */
			dec_err = grl_bmpdec_draw_image_funcs[idx](p_dec_info);
		} else
		{
			ret_err = SVG_INVALID_VALUE;
			SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_DRAW_IMAGE_SYNC");
		}

		if (dec_err != SVG_NO_ERROR)
		{
			/* An error occurred while decoding, remember error but continue */
			ret_err = dec_err;
		}

	}
	ctx->write_info_cnt = 0;

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_DRAW_IMAGE_SYNC");

	return ret_err;
}

static void GRL_BMPDEC_dec_tsk(void* exinf)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGUint32 ret_flg = 0;
	SVGInt32 cnt = 0;
	GRL_BMPDEC_dec_info *p_dec_info = NULL;
	SVGBmpDecImageEncoding idx = SVG_BMPDEC_UNKNOWN_ENCODING;
	SVGContextBmpDec *ctx = NULL;
	SVGBoolean do_abort = SVG_FALSE;
	int ercd = 0;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_DEC_TSK");

	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	ctx = (SVGContextBmpDec*)exinf;

	if (NULL != ctx)
	{
		while (SVG_FALSE == do_abort)
		{
			ret_err = SVG_NO_ERROR;
			cnt = 0;
			/* Wait for start trigger */
			ercd
					= GRL_BMPDEC_WAI_FLG( (GRL_BMPDEC_FLG_PTN_START | GRL_BMPDEC_FLG_PTN_ABORT),
							(FLG_WAI_OR | FLG_BITCLR),
							&ret_flg,
							TMO_FEVR );
			if (ercd > GRL_NO_ERROR)//TODO:: find correct posix error codes
			{
				/* The flag is invalid or was deleted, trace an error and exit task */
				ret_err = SVG_BMPDEC_OS_ERROR;
				SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DEC_TSK");
				do_abort = SVG_TRUE;
			}

			if ((ret_flg & GRL_BMPDEC_FLG_PTN_ABORT) != 0)
			{
				do_abort = SVG_TRUE;
			}

			/*
			 * Call the decoder-specific decoding function for each
			 * queued request.
			 */
			while ((cnt < ctx->read_info_cnt) && (SVG_NO_ERROR == ret_err)
					&& (SVG_FALSE == do_abort))
			{
				/* --Begin critical section-- (protected access to decoding queue) */
				ret_err = GRL_wai_lock_tmo(ctx->busy_lock, GRL_BMPDEC_LOCK_TMO );
				if (SVG_NO_ERROR != ret_err)
				{
					/* Timeout, cancel decoding */
					ret_err = SVG_TIMEOUT_ERROR;
					SVG_BMP_E("SVG_TIMEOUT_ERROR IN GRL_BMPDEC_DEC_TSK");
				} else
				{
					p_dec_info = &ctx->p_read_queue[cnt];
					p_dec_info->revert = ctx->rgb_to_bgr;
					idx = p_dec_info->image.type;

					if ((SVG_BMPDEC_UNKNOWN_ENCODING < idx)
							&& (SVG_BMPDEC_ENCODING_MAX_VALUE > idx))
					{
						/* Use image type as index to look up correct decoding function */
						ret_err = grl_bmpdec_draw_image_funcs[idx](p_dec_info);
					} else
					{
						ret_err = SVG_INVALID_VALUE;
						SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_DEC_TSK");
					}
				}

				cnt++;
				if ((ctx->read_info_cnt <= cnt) || (SVG_NO_ERROR != ret_err))
				{
					/* This was the last entry or error occurred, reset the queue size */
					ctx->read_info_cnt = 0;
					ctx->queue_running = SVG_FALSE;

					/* Inform other task that decoding is finished */
					ercd = GRL_BMPDEC_SET_FLG( GRL_BMPDEC_FLG_PTN_FINISHED );
					if (ercd > GRL_NO_ERROR)//TODO:: find correct posix error codes
					{
						/* The flag is invalid or was deleted, trace an error and exit task */
						SVG_BMP_E("GRL_BMPDEC_DEC_TSK : Flag error while waiting (Invalid ID, deleted or timeout occurred)");
						do_abort = SVG_TRUE;
					}

				}

				/* --End critical section-- (protected access to decoding queue) */
				GRL_sig_lock(ctx->busy_lock);

				/* Check if decoding needs to be aborted (but don't wait) */
				ercd = GRL_BMPDEC_WAI_FLG( GRL_BMPDEC_FLG_PTN_ABORT,
						FLG_WAI_OR |FLG_BITCLR,
						&ret_flg,
						TMO_POL );
				if (ercd > GRL_NO_ERROR)//TODO:: find correct posix error codes
				{
					/* The flag is invalid or was deleted, trace an error and exit task */
					SVG_BMP_E("GRL_BMPDEC_DEC_TSK : Flag error while waiting (Invalid ID, deleted or timeout occurred)");
					do_abort = SVG_TRUE;
				}
				if ((ret_flg & GRL_BMPDEC_FLG_PTN_ABORT) != 0)
				{
					do_abort = SVG_TRUE;
				}
			}
		}
	} else
	{
		SVG_BMP_E("SVG_CONTEXT_NULL IN GRL_BMPDEC_DEC_TSK");
	}

	SVG_BMP_U("EXIT FROM  GRL_BMPDEC_DEC_TSK");

	/* Exit and delete this task */
	GRL_BMPDEC_EXT_TSK;

}

SVGError GRL_BMPDEC_cre_dec_tsk(SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;

	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CRE_DEC_TSK");

	/* Create event flag for status synchronization with decoder task */

	GRL_BMPDEC_CRE_FLG( ctx->dec_flg, ret_err );

	if (0 < ret_err)
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CRE_DEC_TSK");
	}

	if (SVG_NO_ERROR == ret_err)
	{
		GRL_BMPDEC_get_conf(&config);

		/* Create and start decoder task */

		ret_err = GRL_BMPDEC_CRS_TSK( &(ctx->dec_tsk_id),NULL,
				(void*)GRL_BMPDEC_dec_tsk,
				(void*)ctx );
		if (0 != ret_err)
		{
			ctx->dec_tsk_id = 0;
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CRE_DEC_TSK");
		}
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CRE_DEC_TSK");

	return ret_err;
}

void GRL_BMPDEC_swap_queues(SVGContextBmpDec *ctx)
{
	GRL_BMPDEC_dec_info *p_tmp_queue = NULL;
	SVGInt32 tmp_info_cnt = 0;

	p_tmp_queue = ctx->p_write_queue;
	ctx->p_write_queue = ctx->p_read_queue;
	ctx->p_read_queue = p_tmp_queue;

	tmp_info_cnt = ctx->write_info_cnt;
	ctx->write_info_cnt = ctx->read_info_cnt;
	ctx->read_info_cnt = tmp_info_cnt;
}

SVGError GRL_BMPDEC_draw_image_async(SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGUint32 ret_flg = 0;
	SVGUint32 cnt = 0;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};
	int ercd = 0;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_DRAW_IMAGE_ASYNC");

	GRL_BMPDEC_get_conf(&config);

	/* If not already present, the decoder task has to be created (usually when executing the first flush) */
	if (0 == ctx->dec_tsk_id)
	{
		ret_err = GRL_BMPDEC_cre_dec_tsk(ctx);
		if (SVG_NO_ERROR != ret_err)
		{
			SVG_BMP_E("GRL_BMPDEC_DRAW_IMAGE_ASYNC:Could not create decoding task, flushing aborted.");
		}
	}

	/* --Begin critical section-- (protected access to decoding queue) */
	if (SVG_NO_ERROR == ret_err)
	{
		ret_err = GRL_wai_lock_tmo(ctx->busy_lock, GRL_BMPDEC_LOCK_TMO );
		if (SVG_NO_ERROR != ret_err)
		{
			SVG_BMP_E("GRL_BMPDEC_DRAW_IMAGE_ASYNC:Could not aquire busy lock of context, flushing aborted.");
		}
	}

	if (SVG_NO_ERROR == ret_err)
	{
		/*
		 * Check if decoder task is still working on the other queue.
		 * If yes, the current write queue has to be appended to the
		 * currently decoding read queue. This is safe because the
		 * decoder task has to wait for the busy_lock before it can continue.
		 */
		if (ctx->queue_running == SVG_TRUE)
		{
			/* Decoder task is busy, append all pending requests if queue size is big enough */
			if ((ctx->read_info_cnt) + (ctx->write_info_cnt)
					<= (SVGInt32) config.queue_size)
			{
				for (cnt = 0; cnt < (SVGUint32) ctx->write_info_cnt; cnt++)
				{
					ctx->p_read_queue[ctx->read_info_cnt]
							= ctx->p_write_queue[cnt];
					ctx->read_info_cnt++;
				}
				ctx->write_info_cnt = 0;
			} else
			{
				/* Pending requests don't fit into one queue. Therefore it's necessary
				 * to wait until decoding of the current read_queue is finished.
				 *
				 * --End critical section-- (protected access to decoding queue) */
				GRL_sig_lock(ctx->busy_lock);
				ercd = GRL_BMPDEC_WAI_FLG( GRL_BMPDEC_FLG_PTN_FINISHED,
						(FLG_WAI_AND | FLG_BITCLR),
						&ret_flg,
						GRL_BMPDEC_FINISH_TMO );
				if (0 != ercd)
				{
					/* The flag is invalid, was deleted or a timeout occurred */
					ret_err = SVG_BMPDEC_OS_ERROR;
					SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_DRAW_IMAGE_ASYNC");
				} else
				{
					/* --Begin critical section-- (protected access to decoding queue) */
					ret_err = GRL_wai_lock_tmo( ctx->busy_lock,
							GRL_BMPDEC_LOCK_TMO );
					if (SVG_NO_ERROR != ret_err)
					{
						SVG_BMP_E("GRL_BMPDEC_DRAW_IMAGE_ASYNC : Could not aquire busy lock of context, flushing aborted.");
					}
					/* Now the decoder task is idle, swap the queues */
					GRL_BMPDEC_swap_queues(ctx);
				}
			}
		} else
		{
			/* Decoder task is idle, just swap the queues */
			GRL_BMPDEC_swap_queues(ctx);

			/* Mark the read queue as running and set the START flag to
			 * trigger the decoding task
			 */
			ctx->queue_running = SVG_TRUE;
			GRL_BMPDEC_SET_FLG( GRL_BMPDEC_FLG_PTN_START );
		}
	}

	/* --End critical section-- (protected access to decoding queue) */
	GRL_sig_lock(ctx->busy_lock);

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_DRAW_IMAGE_ASYNC");

	return ret_err;
}

SVGError GRL_BMPDEC_wai_dec(SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	int ercd = 0;
	SVGUint32 ret_ptn = 0;
	if (ctx->queue_running == SVG_TRUE)
	{
		/* Wait for the finished flag to be set. */
		ercd = GRL_BMPDEC_WAI_FLG( GRL_BMPDEC_FLG_PTN_FINISHED,
				(FLG_WAI_AND| FLG_BITCLR),
				&ret_ptn,
				GRL_BMPDEC_FINISH_TMO );
		if (0 != ercd)
		{
			/* The flag is invalid, was deleted or a timeout occurred */
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_WAI_DEC");
		}
	}
	return ret_err;
}

SVGError GRL_BMPDEC_get_error(SVGContextBmpDec *ctx)
{
	SVGError temp_error = SVG_NO_ERROR;
	SVGBoolean is_valid = SVG_FALSE;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_GET_ERROR");

	if (NULL == ctx)
	{
		temp_error = g_bmpdec_lasterror;
		g_bmpdec_lasterror = SVG_NO_ERROR;
	} else
	{
		/* If context is not valid return global error instead */
		is_valid = GRL_BMPDEC_clst_is_valid(ctx);

		if (SVG_FALSE == is_valid)
		{
			temp_error = g_bmpdec_lasterror;
			g_bmpdec_lasterror = SVG_NO_ERROR;
		} else
		{
			temp_error = ctx->lastError;
			ctx->lastError = SVG_NO_ERROR;
		}
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_GET_ERROR");

	return temp_error;
}

void GRL_BMPDEC_set_error(SVGContextBmpDec *ctx, SVGError error)
{

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_SET_ERROR");

	if (NULL == ctx)
	{
		if (SVG_NO_ERROR == g_bmpdec_lasterror)
		{
			g_bmpdec_lasterror = error;
		} else
		{
			/* nothing to do */
		}
	} else
	{
		if (SVG_NO_ERROR == ctx->lastError)
		{
			ctx->lastError = error;
		} else
		{
			/* nothing to do */
		}
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_SET_ERROR");
}

void GRL_BMPDEC_clr_error(SVGContextBmpDec *ctx)
{

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CLR_ERROR");

	if (NULL == ctx)
	{
		g_bmpdec_lasterror = SVG_NO_ERROR;
	} else
	{
		ctx->lastError = SVG_NO_ERROR;
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CLR_ERROR");
}

SVGError GRL_BMPDEC_clst_init(void)
{
	SVGError ret_err = SVG_NO_ERROR;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CLST_INIT");

	/* Create lock for accessing the global context counter */
	ret_err = GRL_cre_lock(&g_bmpdec_ctx_lock, GRL_BMPDEC_LCK_NM_CTX );
	if (SVG_NO_ERROR != ret_err)
	{

		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_INIT");
	} else
	{
		GRL_BMPDEC_get_conf(&config);

		/* Allocate memory for context list from 1D resources */
		g_bmpdec_ctx_list = (SVGContextBmpDec**) GRL_malloc_1D_resource(
				(SVGUint32) config.ctx_max * sizeof(SVGContextBmpDec*));
		if (g_bmpdec_ctx_list == NULL)
		{
			ret_err = SVG_OUT_OF_MEMORY;
			SVG_BMP_E("SVG_OUT_OF_MEMORY IN GRL_BMPDEC_CLST_INIT");
			GRL_del_lock(g_bmpdec_ctx_lock);
		} else
		{
			/* Initialize list by setting all entries to 0 */
			memset(g_bmpdec_ctx_list, 0, (SVGUint32) config.ctx_max
					* sizeof(SVGContextBmpDec*));
		}
	}
	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CLST_INIT");

	return ret_err;
}

SVGError GRL_BMPDEC_clst_uninit(void)
{
	SVGError ret_err = SVG_NO_ERROR;

	SVG_BMP_U("ENTER INTO GRL_BMPDEC_CLST_UNINIT");

	/* Free memory for context list */
	if (NULL != g_bmpdec_ctx_list)
	{
		GRL_free_1D_resource(g_bmpdec_ctx_list);
	}

	/* Delete lock */
	ret_err = GRL_del_lock(g_bmpdec_ctx_lock);
	if (SVG_NO_ERROR != ret_err)
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_UNINIT");
	}

	SVG_BMP_U("EXIT FROM GRL_BMPDEC_CLST_UNINIT");

	return ret_err;
}

SVGError GRL_BMPDEC_clst_add(SVGContextBmpDec *new_ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGUint32 ctx_cnt = 0;
	GRL_BMPDEC_conf config  = {0,0,0,0,0,0,0,0,0,0};

	GRL_BMPDEC_get_conf(&config);

	/* --Begin critical section-- (protected access to global context list) */
	ret_err = GRL_wai_lock_tmo(g_bmpdec_ctx_lock, GRL_BMPDEC_LOCK_TMO );
	if (SVG_NO_ERROR == ret_err)
	{
		if (g_bmpdec_ctx_cnt < config.ctx_max)
		{

			/* Find first empty slot */
			ctx_cnt = 0;
			while ((ctx_cnt < config.ctx_max) && (NULL
					!= g_bmpdec_ctx_list[ctx_cnt]))
			{
				ctx_cnt++;
			}

			/* Add the context */
			if (ctx_cnt < config.ctx_max)
			{
				g_bmpdec_ctx_list[ctx_cnt] = new_ctx;
				g_bmpdec_ctx_cnt++;
			} else
			{
				/* List is full */
				ret_err = SVG_BMPDEC_CTX_MAX;
				SVG_BMP_E("SVG_BMPDEC_CTX_MAX IN GRL_BMPDEC_CLST_ADD");
			}
		} else
		{
			/* List is full */
			ret_err = SVG_BMPDEC_CTX_MAX;
			SVG_BMP_E("SVG_BMPDEC_CTX_MAX IN GRL_BMPDEC_CLST_ADD");
		}

		ret_err = GRL_sig_lock(g_bmpdec_ctx_lock);
		if (SVG_NO_ERROR != ret_err)
		{
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_ADD");
		}
	} else
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_ADD");
	}
	/* --End critical section-- (protected access to global context list) */

	return ret_err;
}

SVGError GRL_BMPDEC_clst_del(const SVGContextBmpDec *del_ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGError tmp_err = SVG_NO_ERROR;
	SVGUint32 ctx_cnt = 0;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};

	GRL_BMPDEC_get_conf(&config);

	/* --Begin critical section-- (protected access to global context list) */
	ret_err = GRL_wai_lock_tmo(g_bmpdec_ctx_lock, GRL_BMPDEC_LOCK_TMO );
	if (SVG_NO_ERROR == ret_err)
	{
		/* Find context in list */
		ctx_cnt = 0;
		while ((ctx_cnt < config.ctx_max) && (del_ctx
				!= g_bmpdec_ctx_list[ctx_cnt]))
		{
			ctx_cnt++;
		}

		/* Remove the context */
		if (ctx_cnt < config.ctx_max)
		{
			g_bmpdec_ctx_list[ctx_cnt] = NULL;
			g_bmpdec_ctx_cnt--;
		} else
		{
			/* Not found, the context is invalid */
			ret_err = SVG_BMPDEC_INVALID_CONTEXT;
			SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN GRL_BMPDEC_CLST_DEL");
		}

		tmp_err = GRL_sig_lock(g_bmpdec_ctx_lock);
		if (SVG_NO_ERROR != tmp_err)
		{
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_DEL");
		}
	} else
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_DEL");
	}
	/* --End critical section-- (protected access to global context list) */

	return ret_err;
}

SVGContextBmpDec* GRL_BMPDEC_clst_get_ctx(SVGUint32 ctx_idx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGContextBmpDec *ret_ctx = NULL;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};

	GRL_BMPDEC_get_conf(&config);

	/* --Begin critical section-- (protected access to global context list) */
	ret_err = GRL_wai_lock_tmo(g_bmpdec_ctx_lock, GRL_BMPDEC_LOCK_TMO );
	if (SVG_NO_ERROR == ret_err)
	{
		if (ctx_idx < config.ctx_max)
		{
			ret_ctx = g_bmpdec_ctx_list[ctx_idx];
		} else
		{
			/* Out of range */
			SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN GRL_BMPDEC_CLST_GET_CTX");
			ret_ctx = NULL;
		}

		ret_err = GRL_sig_lock(g_bmpdec_ctx_lock);
		if (SVG_NO_ERROR != ret_err)
		{
			ret_err = SVG_BMPDEC_OS_ERROR;
			SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_GET_CTX");
		}
	} else
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_GET_CTX");
	}
	/* --End critical section-- (protected access to global context list) */

	return ret_ctx;
}

SVGBoolean GRL_BMPDEC_clst_is_valid(const SVGContextBmpDec *ctx)
{
	SVGError ret_err = SVG_NO_ERROR;
	SVGBoolean is_valid = SVG_FALSE;
	SVGUint32 ctx_cnt = 0;
	GRL_BMPDEC_conf config = {0,0,0,0,0,0,0,0,0,0};

	GRL_BMPDEC_get_conf(&config);

	/* --Begin critical section-- (protected access to global context list) */
	ret_err = GRL_wai_lock_tmo(g_bmpdec_ctx_lock, GRL_BMPDEC_LOCK_TMO );
	if (SVG_NO_ERROR == ret_err)
	{
		/* Find context in list */
		while ((ctx_cnt < config.ctx_max)
				&& (ctx != g_bmpdec_ctx_list[ctx_cnt]))
		{
			ctx_cnt++;
		}

		/* Context was found in list */
		if (ctx_cnt < config.ctx_max)
		{
			is_valid = SVG_TRUE;
		} else
		{
			is_valid = SVG_FALSE;
		}

		ret_err = GRL_sig_lock(g_bmpdec_ctx_lock);
		if (SVG_NO_ERROR != ret_err)
		{
			SVG_BMP_E("GRL_BMPDEC_CLST_GET_CTX : Cannot signal lock "
					"context creation/destruction will be continued.");
		}
	} else
	{
		ret_err = SVG_BMPDEC_OS_ERROR;
		SVG_BMP_E("SVG_BMPDEC_OS_ERROR IN GRL_BMPDEC_CLST_GET_CTX ");
	}
	/* --End critical section-- (protected access to global context list) */

	return is_valid;
}

SVGUint32 GRL_BMPDEC_clst_get_num(void)
{
	return (SVGUint32) g_bmpdec_ctx_cnt;
}
